﻿namespace Microsoft.Samples.PlanMyNight.Data.Caching
{
    using System;
    using System.Collections.Generic;
    using System.Diagnostics;
    using System.IO;
    using System.Linq;
    using System.Runtime.Serialization.Json;
    using System.Text;
    using System.Threading.Tasks;
    using Microsoft.Samples.PlanMyNight.Entities;

    public class CachedActivitiesRepository : IActivitiesRepository
    {
        private const string ReferenceDataCacheContainer = "Reference";

        private const string GeocodeCacheContainer = "Geocode";

        private const string ActivityCacheContainer = "Activity";

        private const string SearchQueryCacheContainer = "ActivitySearch";

        private readonly TimeSpan ReferenceDataExpiration = TimeSpan.FromDays(1);

        private readonly TimeSpan GeocodeCacheExpiration = TimeSpan.FromMinutes(180);

        private readonly TimeSpan ActivityCacheExpiration = TimeSpan.FromMinutes(30);

        private readonly TimeSpan SearchQueryCacheExpiration = TimeSpan.FromMinutes(30);

        private readonly ICachingProvider cacheProvider;

        private readonly IActivitiesRepository repository;

        public CachedActivitiesRepository(ICachingProvider cacheProvider) :
            this(cacheProvider, new BingActivitiesRepository())
        {
        }

        public CachedActivitiesRepository(ICachingProvider cacheProvider, IActivitiesRepository repository)
        {
            this.cacheProvider = cacheProvider;
            this.repository = repository;
        }

        public Activity RetrieveActivity(string id)
        {
            var rawKey = Encoding.Default.GetString(Convert.FromBase64String(id)).Split('|');
            var cacheKey = rawKey[0];

            Activity data = this.cacheProvider.Get(ActivityCacheContainer, cacheKey) as Activity;
            if (data != null)
            {
                Trace.WriteLine("GET-CACHE:" + ActivityCacheContainer + "(" + cacheKey + ")");
                return data;
            }

            data = this.repository.RetrieveActivity(id);
            this.cacheProvider.Add(ActivityCacheContainer, cacheKey, data, this.ActivityCacheExpiration);
            Trace.WriteLine("ADD-CACHE:" + ActivityCacheContainer + "(" + cacheKey + ")");
            return data;
        }

        public PagingResult<Activity> Search(AdvancedSearchQuery searchCriteria)
        {
            var cacheKey = Serialize(searchCriteria);

            PagingResult<Activity> data = this.cacheProvider.Get(SearchQueryCacheContainer, cacheKey) as PagingResult<Activity>;
            if (data != null)
            {
                Trace.WriteLine("GET-CACHE:" + SearchQueryCacheContainer + "(" + cacheKey + ")");
                return data;
            }

            data = this.repository.Search(searchCriteria);
            Trace.WriteLine("ADD-CACHE:" + SearchQueryCacheContainer + "(" + cacheKey + ")");
            this.cacheProvider.Add(SearchQueryCacheContainer, cacheKey, data, this.SearchQueryCacheExpiration);

            // cache activities
            foreach (var activity in data.Items)
            {
                if (!this.cacheProvider.HasKey(ActivityCacheContainer, activity.BingId))
                {
                    Trace.WriteLine("ADD-CACHE:" + ActivityCacheContainer + "(" + activity.BingId + ")");
                    this.cacheProvider.Add(ActivityCacheContainer, activity.BingId, activity, this.ActivityCacheExpiration);
                }
            }

            return data;
        }

        public PagingResult<Activity> Search(NaturalSearchQuery query)
        {
            var cacheKey = Serialize(query);

            PagingResult<Activity> data = this.cacheProvider.Get(SearchQueryCacheContainer, cacheKey) as PagingResult<Activity>;
            if (data != null)
            {
                Trace.WriteLine("GET-CACHE:" + SearchQueryCacheContainer + "(" + cacheKey + ")");
                return data;
            }

            data = this.repository.Search(query);
            Trace.WriteLine("ADD-CACHE:" + SearchQueryCacheContainer + "(" + cacheKey + ")");
            this.cacheProvider.Add(SearchQueryCacheContainer, cacheKey, data, this.SearchQueryCacheExpiration);

            // cache activities
            foreach (var activity in data.Items)
            {
                if (!this.cacheProvider.HasKey(ActivityCacheContainer, activity.BingId))
                {
                    Trace.WriteLine("ADD-CACHE:" + ActivityCacheContainer + "(" + activity.BingId + ")");
                    this.cacheProvider.Add(ActivityCacheContainer, activity.BingId, activity, this.ActivityCacheExpiration);
                }
            }

            return data;
        }

        public ActivityAddress ParseQueryLocation(string query)
        {
            var cacheKey = query.ToUpperInvariant();
            
            ActivityAddress data = this.cacheProvider.Get(GeocodeCacheContainer, cacheKey) as ActivityAddress;
            if (data != null)
            {
                Trace.WriteLine("GET-CACHE:" + GeocodeCacheContainer + "(" + cacheKey + ")");
                return data;
            }

            data = this.repository.ParseQueryLocation(query);
            Trace.WriteLine("ADD-CACHE:" + GeocodeCacheContainer + "(" + cacheKey + ")");
            this.cacheProvider.Add(GeocodeCacheContainer, cacheKey, data, this.GeocodeCacheExpiration);

            return data;
        }

        public Tuple<double, double> GeocodeAddress(ActivityAddress address)
        {
            var cacheKey = Serialize(address);

            Tuple<double, double> data = this.cacheProvider.Get(GeocodeCacheContainer, cacheKey) as Tuple<double, double>;
            if (data != null)
            {
                Trace.WriteLine("GET-CACHE:" + GeocodeCacheContainer + "(" + cacheKey + ")");
                return data;
            }

            data = this.repository.GeocodeAddress(address);
            this.cacheProvider.Add(GeocodeCacheContainer, cacheKey, data, this.GeocodeCacheExpiration);
            Trace.WriteLine("ADD-CACHE:" + GeocodeCacheContainer + "(" + cacheKey + ")");
            return data;
        }

        public void PopulateItineraryActivities(Itinerary itinerary)
        {
            // Sequencial retrieval
            ////foreach (var item in itinerary.Activities.Where(i => i.Activity == null))
            ////{
            ////    item.Activity = this.RetrieveActivity(item.ActivityId);
            ////}

            // Parallel retrieval
            Parallel.ForEach(itinerary.Activities.Where(i => i.Activity == null),
                item =>
                {
                    item.Activity = this.RetrieveActivity(item.ActivityId);
                });
        }

        public IEnumerable<ActivityType> RetrieveActivityTypes()
        {
            var cacheKey = "activityTypes";

            IEnumerable<ActivityType> data = this.cacheProvider.Get(ReferenceDataCacheContainer, cacheKey) as IEnumerable<ActivityType>;
            if (data != null)
            {
                Trace.WriteLine("GET-CACHE:" + ReferenceDataCacheContainer + "(" + cacheKey + ")");
                return data;
            }

            data = this.repository.RetrieveActivityTypes();
            this.cacheProvider.Add(ReferenceDataCacheContainer, cacheKey, data, this.ReferenceDataExpiration);
            Trace.WriteLine("ADD-CACHE:" + ReferenceDataCacheContainer + "(" + cacheKey + ")");
            return data;
        }

        private static string Serialize<T>(T keyObject)
        {
            using (var stream = new MemoryStream())
            {
                var serializer = new DataContractJsonSerializer(typeof(T));
                serializer.WriteObject(stream, keyObject);
                return Encoding.Default.GetString(stream.ToArray());
            }
        }
    }
}
